<?php
if ( ! defined( 'ABSPATH' ) ) exit;

class SGPDF_Admin {
	public function __construct() {
		add_action('admin_menu', array($this,'register_menu'));
		add_action('admin_init', array($this,'register_settings'));
		// legacy imports
		add_action( 'admin_post_sgpdfm_scan_import', array( $this, 'handle_scan_import' ) );
		add_action( 'admin_post_sgpdfm_import_ids', array( $this, 'handle_import_ids' ) );
		// migrate
		add_action( 'admin_post_sgpdfm_migrate', array( $this, 'handle_migrate' ) );
	}

	public function register_menu() {
		// Top level
		add_menu_page( __('sgPDF Manager','sgpdf-manager'), __('sgPDF Manager','sgpdf-manager'),
			'manage_options','sgpdfm', array($this,'render_dashboard'), 'dashicons-media-document', 58 );

		// Submenus
		add_submenu_page('sgpdfm', __('PDF / Documents','sgpdf-manager'), __('PDF / Documents','sgpdf-manager'),
			'manage_options','edit.php?post_type=sgpdf_doc' );
		add_submenu_page('sgpdfm', __('Add New / Edit','sgpdf-manager'), __('Add New / Edit','sgpdf-manager'),
			'manage_options','post-new.php?post_type=sgpdf_doc' );
		add_submenu_page('sgpdfm', __('Add by FTP','sgpdf-manager'), __('Add by FTP','sgpdf-manager'),
			'manage_options','sgpdfm-ftp', array($this,'render_add_by_ftp') );
		add_submenu_page('sgpdfm', __('Add by Media Library','sgpdf-manager'), __('Add by Media Library','sgpdf-manager'),
			'manage_options','sgpdfm-media', array($this,'render_add_by_media') );
		add_submenu_page('sgpdfm', __('Categories','sgpdf-manager'), __('Categories','sgpdf-manager'),
			'manage_options','edit-tags.php?taxonomy=sgpdf_category&post_type=sgpdf_doc' );
		add_submenu_page('sgpdfm', __('Tags','sgpdf-manager'), __('Tags','sgpdf-manager'),
			'manage_options','edit-tags.php?taxonomy=sgpdf_tag&post_type=sgpdf_doc' );
		add_submenu_page('sgpdfm', __('Notifications','sgpdf-manager'), __('Notifications','sgpdf-manager'),
			'manage_options','sgpdfm-notifications', array($this,'render_notifications') );
		add_submenu_page('sgpdfm', __('Settings','sgpdf-manager'), __('Settings','sgpdf-manager'),
			'manage_options','sgpdfm-settings', array($this,'render_settings') );
		add_submenu_page('sgpdfm', __('Migrate','sgpdf-manager'), __('Migrate','sgpdf-manager'),
			'manage_options','sgpdfm-migrate', array($this,'render_migrate') );
	}

	public function register_settings() {
		register_setting( 'sgpdfm_settings', 'sgpdfm_allowed_types', array(
			'type'=>'array','sanitize_callback'=>array($this,'sanitize_mimes'),
			'default'=>sgpdfm_default_allowed_types()
		));
		register_setting( 'sgpdfm_settings', 'sgpdfm_roles_allowed', array(
			'type'=>'array','sanitize_callback'=>array($this,'sanitize_roles'),'default'=>array('administrator')
		));
		register_setting( 'sgpdfm_settings', 'sgpdfm_featured_enabled', array('type'=>'boolean','default'=>false) );
		register_setting( 'sgpdfm_settings', 'sgpdfm_front_style', array('type'=>'string','default'=>'light') );
		register_setting( 'sgpdfm_settings', 'sgpdfm_pretty_links', array('type'=>'boolean','default'=>false) );
		register_setting( 'sgpdfm_settings', 'sgpdfm_enable_viewer', array('type'=>'boolean','default'=>false) );
	}

	public function sanitize_mimes($val){
		if (!is_array($val)) return sgpdfm_default_allowed_types();
		$out=array();
		foreach ($val as $m) { foreach ( array_map('trim', explode(',', sanitize_text_field($m))) as $mm ) { if($mm) $out[]=$mm; } }
		return array_values(array_unique($out));
	}
	public function sanitize_roles($val){
		if (!is_array($val)) return array('administrator');
		return array_values(array_unique(array_map('sanitize_text_field',$val)));
	}

	public function render_dashboard() {
		echo '<div class="wrap"><h1>'.esc_html__('sgPDF Manager — Dashboard','sgpdf-manager').'</h1>';
		echo '<p>'.esc_html__('Use the left submenus to manage documents.','sgpdf-manager').'</p></div>';
	}

	/* -------- Settings (Tabbed UI like screenshot) -------- */
	private function settings_tabs() {
		$tabs = array(
			'general'=>__('General settings','sgpdf-manager'),
			'upload'=>__('Upload','sgpdf-manager'),
			'backend'=>__('Backend Access','sgpdf-manager'),
			'featured'=>__('Featured Image','sgpdf-manager'),
			'styles'=>__('Styles','sgpdf-manager'),
			'permalinks'=>__('Permalinks','sgpdf-manager'),
			'viewer'=>__('Embeded Viewer','sgpdf-manager'),
		);
		$active = isset($_GET['tab']) ? sanitize_key($_GET['tab']) : 'general';
		echo '<h1>'.esc_html__('BSK PDF Settings','sgpdf-manager').'</h1>';
		echo '<h2 class="nav-tab-wrapper">';
		foreach ($tabs as $key=>$label) {
			$class = ($active===$key) ? ' nav-tab nav-tab-active' : ' nav-tab';
			echo '<a class="'.$class.'" href="'.esc_url( admin_url('admin.php?page=sgpdfm-settings&tab='.$key) ).'">'.esc_html($label).'</a>';
		}
		echo '</h2>';
		return $active;
	}

	public function render_settings() {
		if ( ! current_user_can( 'manage_options' ) ) wp_die('Forbidden');
		$tab = $this->settings_tabs();
		echo '<div class="sgpdfm-settings-panel">';
		switch ($tab) {
			case 'general':
				$this->settings_general(); break;
			case 'upload':
				$this->settings_upload(); break;
			case 'backend':
				$this->settings_backend(); break;
			case 'featured':
				$this->settings_featured(); break;
			case 'styles':
				$this->settings_styles(); break;
			case 'permalinks':
				$this->settings_permalinks(); break;
			case 'viewer':
				$this->settings_viewer(); break;
		}
		echo '</div>';
	}

	private function settings_general() { ?>
		<form method="post" action="options.php">
			<?php settings_fields('sgpdfm_settings'); ?>
			<h2><?php esc_html_e('General settings','sgpdf-manager'); ?></h2>
			<p><?php esc_html_e('Core behaviour for shortcodes and UI can be customized here.','sgpdf-manager'); ?></p>
			<table class="form-table">
				<tr>
					<th><?php esc_html_e('Front-end style scheme','sgpdf-manager'); ?></th>
					<td>
						<select name="sgpdfm_front_style">
							<?php $v = get_option('sgpdfm_front_style','light'); ?>
							<option value="light" <?php selected($v,'light'); ?>>Light</option>
							<option value="dark" <?php selected($v,'dark'); ?>>Dark</option>
							<option value="none" <?php selected($v,'none'); ?>>None</option>
						</select>
					</td>
				</tr>
			</table>
			<?php submit_button(); ?>
		</form>
	<?php }

	private function settings_upload() {
		$allowed = sgpdfm_get_allowed_types();
		$mime_choices = array(
			'application/pdf'=>'PDF (.pdf)',
			'application/vnd.openxmlformats-officedocument.wordprocessingml.document'=>'DOCX (.docx)',
			'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'=>'XLSX (.xlsx)',
			'application/vnd.openxmlformats-officedocument.presentationml.presentation'=>'PPTX (.pptx)',
			'text/plain'=>'TXT (.txt)','text/csv'=>'CSV (.csv)','application/zip'=>'ZIP (.zip)',
		); ?>
		<form method="post" action="options.php">
			<?php settings_fields('sgpdfm_settings'); ?>
			<h2><?php esc_html_e('Upload','sgpdf-manager'); ?></h2>
			<table class="form-table"><tbody>
				<?php foreach ($mime_choices as $mime=>$label): ?>
				<tr><th><?php echo esc_html($label); ?></th><td>
					<label><input type="checkbox" name="sgpdfm_allowed_types[]" value="<?php echo esc_attr($mime); ?>" <?php checked(in_array($mime,$allowed,true)); ?> /> <?php echo esc_html($mime); ?></label>
				</td></tr>
				<?php endforeach; ?>
				<tr><th><?php esc_html_e('Custom MIME types','sgpdf-manager'); ?></th><td>
					<input type="text" name="sgpdfm_allowed_types[]" class="regular-text" placeholder="application/msword, image/svg+xml" />
				</td></tr>
			</tbody></table>
			<?php submit_button( __('Save Upload Settings','sgpdf-manager') ); ?>
		</form>
	<?php }

	private function settings_backend() {
		$roles = wp_roles()->roles;
		$allowed = (array) get_option('sgpdfm_roles_allowed', array('administrator')); ?>
		<form method="post" action="options.php">
			<?php settings_fields('sgpdfm_settings'); ?>
			<h2><?php esc_html_e('Backend Access','sgpdf-manager'); ?></h2>
			<table class="form-table"><tbody>
				<tr><th><?php esc_html_e('Roles allowed in sgPDF Manager','sgpdf-manager'); ?></th><td>
					<?php foreach ($roles as $key=>$role): ?>
						<label style="display:inline-block;margin-right:12px;"><input type="checkbox" name="sgpdfm_roles_allowed[]" value="<?php echo esc_attr($key); ?>" <?php checked(in_array($key,$allowed,true)); ?> /> <?php echo esc_html($role['name']); ?></label>
					<?php endforeach; ?>
				</td></tr>
			</tbody></table>
			<?php submit_button( __('Save Access Settings','sgpdf-manager') ); ?>
		</form>
	<?php }

	private function settings_featured() { ?>
		<form method="post" action="options.php">
			<?php settings_fields('sgpdfm_settings'); ?>
			<h2><?php esc_html_e('Featured Image','sgpdf-manager'); ?></h2>
			<label><input type="checkbox" name="sgpdfm_featured_enabled" value="1" <?php checked( (bool) get_option('sgpdfm_featured_enabled',false) ); ?> /> <?php esc_html_e('Enable featured image for documents','sgpdf-manager'); ?></label>
			<?php submit_button( __('Save Featured Image Settings','sgpdf-manager') ); ?>
		</form>
	<?php }

	private function settings_styles() { ?>
		<div class="wrap">
			<h2><?php esc_html_e('Styles','sgpdf-manager'); ?></h2>
			<p><?php esc_html_e('Front-end styles are minimal; override via your theme or custom CSS.','sgpdf-manager'); ?></p>
		</div>
	<?php }

	private function settings_permalinks() { ?>
		<form method="post" action="options.php">
			<?php settings_fields('sgpdfm_settings'); ?>
			<h2><?php esc_html_e('Permalinks','sgpdf-manager'); ?></h2>
			<label><input type="checkbox" name="sgpdfm_pretty_links" value="1" <?php checked( (bool) get_option('sgpdfm_pretty_links',false) ); ?> /> <?php esc_html_e('Enable pretty permalinks for document links','sgpdf-manager'); ?></label>
			<?php submit_button( __('Save Permalink Settings','sgpdf-manager') ); ?>
		</form>
	<?php }

	private function settings_viewer() { ?>
		<div class="wrap">
			<h2><?php esc_html_e('Embedded PDF Viewer','sgpdf-manager'); ?></h2>
			<p><?php echo wp_kses_post( sprintf( __('%1$s helps you to use %2$s to display PDF content.', 'sgpdf-manager'),
				'<strong>sgPDF Manager</strong>', '<a href="https://mozilla.github.io/pdf.js/" target="_blank" rel="noopener">Mozilla\'s PDF.js</a>' ) ); ?></p>
			<p><?php echo wp_kses_post( sprintf( __('Mozilla\'s PDF.js is licensed under the Apache License 2.0 and you may check its license from %s.', 'sgpdf-manager'),
				'<a href="https://github.com/mozilla/pdf.js/blob/master/LICENSE" target="_blank" rel="noopener">GitHub</a>' ) ); ?></p>
			<form method="post" action="options.php">
				<?php settings_fields('sgpdfm_settings'); ?>
				<label><input type="checkbox" name="sgpdfm_enable_viewer" value="1" <?php checked( (bool) get_option('sgpdfm_enable_viewer',false) ); ?> /> <?php esc_html_e('Enable Embedded Viewer','sgpdf-manager'); ?></label>
				<?php submit_button( __('Save Embedded Viewer Settings','sgpdf-manager') ); ?>
			</form>
			<p class="description"><?php esc_html_e('When enabled, shortcodes can use viewer="true" to open documents in an embedded PDF.js viewer.', 'sgpdf-manager'); ?></p>
		</div>
	<?php }

	/* ------- Add by FTP / Media / Migrate (same as v3.0) ------- */
	public function render_add_by_ftp() {
		if ( ! current_user_can( 'manage_options' ) ) wp_die('Forbidden');
		$upload_dir = wp_upload_dir();
		$import_dir = trailingslashit( $upload_dir['basedir'] ) . 'sgpdf-import';
		if ( ! file_exists( $import_dir ) ) wp_mkdir_p( $import_dir );
		?>
		<div class="wrap">
			<h1><?php esc_html_e('Add by FTP','sgpdf-manager'); ?></h1>
			<p><?php printf( esc_html__('Place files in %s then click Import.','sgpdf-manager'), '<code>'.esc_html(str_replace(ABSPATH,'',$import_dir)).'</code>' ); ?></p>
			<form method="post" action="<?php echo esc_url( admin_url('admin-post.php') ); ?>">
				<?php wp_nonce_field('sgpdfm_scan_import','sgpdfm_nonce'); ?>
				<input type="hidden" name="action" value="sgpdfm_scan_import" />
				<p>
					<label><?php esc_html_e('Assign Category','sgpdf-manager'); ?></label>
					<?php wp_dropdown_categories(array('taxonomy'=>'sgpdf_category','hide_empty'=>false,'name'=>'sgpdfm_category','show_option_all'=>__('— None —','sgpdf-manager'))); ?>
				</p>
				<?php submit_button(__('Scan & Import','sgpdf-manager')); ?>
			</form>
		</div>
		<?php
	}

	public function render_add_by_media() {
		if ( ! current_user_can( 'manage_options' ) ) wp_die('Forbidden'); ?>
		<div class="wrap">
			<h1><?php esc_html_e('Add by Media Library','sgpdf-manager'); ?></h1>
			<form method="post" action="<?php echo esc_url( admin_url('admin-post.php') ); ?>">
				<?php wp_nonce_field('sgpdfm_import_ids','sgpdfm_nonce'); ?>
				<input type="hidden" name="action" value="sgpdfm_import_ids" />
				<p><label><?php esc_html_e('Attachment IDs (comma-separated)','sgpdf-manager'); ?></label>
				<input type="text" class="regular-text" name="sgpdfm_ids" placeholder="12,34,56" /></p>
				<p><label><?php esc_html_e('Assign Category','sgpdf-manager'); ?></label>
				<?php wp_dropdown_categories(array('taxonomy'=>'sgpdf_category','hide_empty'=>false,'name'=>'sgpdfm_category','show_option_all'=>__('— None —','sgpdf-manager'))); ?></p>
				<?php submit_button(__('Import','sgpdf-manager')); ?>
			</form>
		</div>
		<?php
	}

	public function render_notifications() {
		echo '<div class="wrap"><h1>'.esc_html__('Notifications','sgpdf-manager').'</h1><p>'.esc_html__('(Placeholder) Configure alert rules here.','sgpdf-manager').'</p></div>';
	}

	public function render_migrate() {
		if ( ! current_user_can( 'manage_options' ) ) wp_die('Forbidden'); ?>
		<div class="wrap">
			<h1><?php esc_html_e('Migrate to v3.x','sgpdf-manager'); ?></h1>
			<p><?php esc_html_e('This will import previously managed attachments (_sgdoc_managed=1) into the new Document post type.','sgpdf-manager'); ?></p>
			<form method="post" action="<?php echo esc_url( admin_url('admin-post.php') ); ?>">
				<?php wp_nonce_field('sgpdfm_migrate','sgpdfm_nonce'); ?>
				<input type="hidden" name="action" value="sgpdfm_migrate" />
				<?php submit_button(__('Run Migration','sgpdf-manager'),'primary'); ?>
			</form>
		</div>
		<?php
	}

	/* Handlers for imports & migration (same logic as v3.0) */
	private function create_doc_from_attachment( $att_id, $category = 0 ) {
		$title = get_the_title( $att_id );
		$doc_id = wp_insert_post(array('post_type'=>'sgpdf_doc','post_status'=>'publish','post_title'=>$title));
		if ( ! is_wp_error($doc_id) ) {
			update_post_meta($doc_id,'_sgpdf_attachment_id',$att_id);
			if ( $category ) wp_set_object_terms($doc_id, array(intval($category)), 'sgpdf_category', false);
			update_post_meta($doc_id,'_sgdoc_managed',1);
		}
		return $doc_id;
	}
	public function handle_scan_import() {
		if ( ! current_user_can( 'manage_options' ) ) wp_die('Forbidden');
		check_admin_referer('sgpdfm_scan_import','sgpdfm_nonce');
		$upload_dir = wp_upload_dir();
		$import_dir = trailingslashit( $upload_dir['basedir'] ) . 'sgpdf-import';
		$cat = isset($_POST['sgpdfm_category']) ? absint($_POST['sgpdfm_category']) : 0;
		$count = 0;
		if ( file_exists($import_dir) ) {
			$files = glob($import_dir.'/*.*');
			require_once ABSPATH . 'wp-admin/includes/file.php';
			foreach ( $files as $file_path ) {
				$filename = basename($file_path);
				$mime = wp_check_filetype($filename)['type'];
				if ( ! $mime || ! in_array($mime, sgpdfm_get_allowed_types(), true) ) continue;
				$bits = wp_upload_bits( $filename, null, @file_get_contents($file_path) );
				if ( ! $bits['error'] ) {
					$att_id = wp_insert_attachment(array(
						'post_mime_type'=>$mime,'post_title'=>pathinfo($filename,PATHINFO_FILENAME),'post_status'=>'inherit'
					), $bits['file']);
					if ( ! is_wp_error($att_id) ) {
						require_once ABSPATH . 'wp-admin/includes/image.php';
						wp_update_attachment_metadata($att_id, wp_generate_attachment_metadata($att_id, $bits['file']));
						$this->create_doc_from_attachment($att_id, $cat);
						$count++; @unlink($file_path);
					}
				}
			}
		}
		wp_redirect( admin_url('admin.php?page=sgpdfm-ftp&sgpdfm_msg=imported_'.$count) ); exit;
	}
	public function handle_import_ids() {
		if ( ! current_user_can( 'manage_options' ) ) wp_die('Forbidden');
		check_admin_referer('sgpdfm_import_ids','sgpdfm_nonce');
		$ids_raw = isset($_POST['sgpdfm_ids']) ? sanitize_text_field( wp_unslash($_POST['sgpdfm_ids']) ) : '';
		$ids = array_filter( array_map('absint', explode(',', $ids_raw)) );
		$cat = isset($_POST['sgpdfm_category']) ? absint($_POST['sgpdfm_category']) : 0;
		$count = 0;
		foreach ($ids as $id) {
			if ( get_post_mime_type($id) ) { $this->create_doc_from_attachment($id, $cat); $count++; }
		}
		wp_redirect( admin_url('admin.php?page=sgpdfm-media&sgpdfm_msg=imported_'.$count) ); exit;
	}
	public function handle_migrate() {
		if ( ! current_user_can( 'manage_options' ) ) wp_die('Forbidden');
		check_admin_referer('sgpdfm_migrate','sgpdfm_nonce');
		$atts = get_posts(array('post_type'=>'attachment','post_status'=>'inherit','numberposts'=>-1,
			'meta_key'=>'_sgdoc_managed','meta_value'=>1));
		$count=0;
		foreach ($atts as $a) {
			$existing = get_posts(array('post_type'=>'sgpdf_doc','meta_key'=>'_sgpdf_attachment_id','meta_value'=>$a->ID,'fields'=>'ids'));
			if ($existing) continue;
			$this->create_doc_from_attachment($a->ID, 0); $count++;
		}
		wp_redirect( admin_url('admin.php?page=sgpdfm-migrate&sgpdfm_msg=migrated_'.$count) ); exit;
	}
}
