<?php
if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * HT5E_Upload
 * Handles uploading, extracting, and registering HTML5/SCORM zip packages
 * Supports: ZipArchive → WP unzip_file → CLI unzip (triple extraction)
 */
class HT5E_Upload {

    public function __construct() {
        add_action( 'wp_ajax_ht5e_upload_zip', [ $this, 'handle_upload' ] );
    }

    /**
     * Locate main entry file inside extracted directory
     */
    private function find_entry( $base ) {
        $targets = [
            'index.html', 'story.html', 'story_html5.html',
            'launch.html', 'presentation.html',
            'index_lms.html', 'course.html', 'main.html'
        ];

        $it = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator( $base, FilesystemIterator::SKIP_DOTS )
        );
        foreach ( $it as $f ) {
            if ( $f->isFile() && in_array( strtolower( $f->getFilename() ), $targets, true ) ) {
                return ltrim( str_replace( $base, '', $f->getPathname() ), '/\\' );
            }
        }
        return '';
    }

    /**
     * Attempt CLI unzip as final fallback
     */
    private function cli_unzip( $zip, $to ) {
        if ( ! function_exists( 'exec' ) ) return false;
        @exec( 'which unzip', $out, $code );
        if ( $code !== 0 || empty( $out[0] ) ) return false;
        $bin = trim( $out[0] );
        @wp_mkdir_p( $to );
        @exec( escapeshellcmd( $bin ) . ' -oq ' . escapeshellarg( $zip ) . ' -d ' . escapeshellarg( $to ), $o, $c );
        return $c === 0;
    }

    /**
     * AJAX handler for zip uploads
     */
    public function handle_upload() {
        check_ajax_referer( 'ht5e-upload', 'nonce' );

        if ( ! current_user_can( 'upload_files' ) )
            wp_send_json_error( 'permission_denied' );

        if ( empty( $_FILES['file'] ) || $_FILES['file']['error'] !== UPLOAD_ERR_OK )
            wp_send_json_error( 'invalid_file' );

        $file  = $_FILES['file'];
        $title = isset( $_POST['ht5e_title'] ) && $_POST['ht5e_title'] !== ''
            ? sanitize_text_field( $_POST['ht5e_title'] )
            : pathinfo( $file['name'], PATHINFO_FILENAME );

        $uploads = wp_upload_dir();
        $base    = trailingslashit( $uploads['basedir'] ) . 'ht5-packages/';
        $urlbase = trailingslashit( $uploads['baseurl'] ) . 'ht5-packages/';
        @wp_mkdir_p( $base );

        // Temporary filename
        $tmpZip = $base . 'tmp_' . wp_generate_password( 8, false, false ) . '.zip';
        if ( ! @move_uploaded_file( $file['tmp_name'], $tmpZip ) )
            wp_send_json_error( 'move_failed' );

        $slug = sanitize_title( $title ) . '-' . time();
        $dest = $base . $slug;
        $work = $base . 'extract_' . wp_generate_password( 6, false, false );
        @wp_mkdir_p( $dest );
        @wp_mkdir_p( $work );

        $ok = false;
        $attempts = [];

        // 1️⃣ ZipArchive
        if ( class_exists( 'ZipArchive' ) ) {
            $z = new ZipArchive();
            if ( $z->open( $tmpZip ) === true ) {
                $ok = @$z->extractTo( $work );
                $z->close();
                $attempts[] = 'ZipArchive:' . ( $ok ? 'ok' : 'fail' );
            } else {
                $attempts[] = 'ZipArchive:not_opened';
            }
        }

        // 2️⃣ WP unzip_file
        if ( ! $ok ) {
            if ( ! function_exists( 'WP_Filesystem' ) )
                require_once ABSPATH . 'wp-admin/includes/file.php';
            WP_Filesystem();
            $res = unzip_file( $tmpZip, $work );
            if ( is_wp_error( $res ) ) {
                $attempts[] = 'WP unzip_file:fail(' . $res->get_error_message() . ')';
            } else {
                $ok = true;
                $attempts[] = 'WP unzip_file:ok';
            }
        }

        // 3️⃣ CLI unzip
        if ( ! $ok ) {
            if ( $this->cli_unzip( $tmpZip, $work ) ) {
                $ok = true;
                $attempts[] = 'CLI unzip:ok';
            } else {
                $attempts[] = 'CLI unzip:fail';
            }
        }

        @unlink( $tmpZip );
        if ( ! $ok )
            wp_send_json_error( 'extract_failed' );

        // Flatten single-directory zips
        $items = array_values( array_diff( scandir( $work ), [ '.', '..' ] ) );
        $root = $work;
        if ( count( $items ) === 1 && is_dir( $work . '/' . $items[0] ) )
            $root = $work . '/' . $items[0];

        $it = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator( $root, FilesystemIterator::SKIP_DOTS ),
            RecursiveIteratorIterator::SELF_FIRST
        );
        foreach ( $it as $node ) {
            $target = $dest . '/' . $it->getSubPathName();
            if ( $node->isDir() ) {
                @mkdir( $target, 0755, true );
            } else {
                @copy( $node->getPathname(), $target );
            }
        }

        $this->rrmdir( $work );

        // Find entry
        $entryRel = $this->find_entry( $dest );
        if ( ! $entryRel )
            wp_send_json_error( 'entry_not_found' );

        $entryUrl = $urlbase . $slug . '/' . str_replace( DIRECTORY_SEPARATOR, '/', $entryRel );

        // Create CPT entry
        $post_id = wp_insert_post( [
            'post_title'  => $title,
            'post_type'   => 'ht5_package',
            'post_status' => 'publish'
        ] );
        update_post_meta( $post_id, '_ht5e_entry_url', esc_url_raw( $entryUrl ) );

        $payload = [
            'id'        => $post_id,
            'entry'     => $entryUrl,
            'shortcode' => '[ht5 id="' . $post_id . '"]',
            'attempts'  => $attempts
        ];
        wp_send_json_success( $payload );
    }

    /**
     * Recursive directory delete
     */
    private function rrmdir( $dir ) {
        if ( ! file_exists( $dir ) ) return;
        $it = new RecursiveDirectoryIterator( $dir, FilesystemIterator::SKIP_DOTS );
        $files = new RecursiveIteratorIterator( $it, RecursiveIteratorIterator::CHILD_FIRST );
        foreach ( $files as $f ) {
            if ( $f->isDir() ) @rmdir( $f->getRealPath() );
            else @unlink( $f->getRealPath() );
        }
        @rmdir( $dir );
    }
}
