<?php
if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * HT5E_Frontend
 * Handles frontend display of HTML5 packages:
 *  - Inline embeds
 *  - Lightbox launch & overlay
 *  - Fullscreen and loader handling
 */
class HT5E_Frontend {

    public function __construct() {
        add_action( 'wp_enqueue_scripts', [ $this, 'register_assets' ] );
        add_action( 'wp_footer', [ $this, 'lightbox_html' ] );   // inject HTML container
        add_action( 'wp_footer', [ $this, 'lightbox_script' ], 20 ); // inject JS
    }

    /**
     * Register and enqueue CSS + JS
     */
    public function register_assets() {
        wp_register_style( 'ht5e-frontend', HT5E_URL . 'assets/css/frontend.css', [], HT5E_VERSION );
        wp_register_script( 'ht5e-frontend', HT5E_URL . 'assets/js/frontend.js', [ 'jquery' ], HT5E_VERSION, true );

        // Load automatically if shortcode exists on page
        if ( is_singular() && has_shortcode( get_post()->post_content ?? '', 'ht5' ) ) {
            wp_enqueue_style( 'ht5e-frontend' );
            wp_enqueue_script( 'ht5e-frontend' );
        }
    }

    /**
     * Inject reusable lightbox HTML (one per page)
     */
    public function lightbox_html() {
        static $printed = false;
        if ( $printed ) return;
        $printed = true;
        ?>
        <div id="ht5e-lightbox" aria-hidden="true" style="display:none;">
            <div class="ht5e-overlay" title="Click to close"></div>
            <div class="ht5e-modal">
                <!-- Close Button -->
                <button class="ht5e-close" aria-label="Close">&times;</button>

                <!-- Loader -->
                <div class="ht5e-loader" id="ht5e-loader" aria-hidden="true">
                    <div class="ht5e-spinner"></div>
                    <div class="ht5e-loading-text">Loading…</div>
                </div>

                <!-- Iframe container -->
                <div class="ht5e-frame">
                    <iframe class="ht5e-iframe" src="" allow="fullscreen" loading="lazy"></iframe>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Inline lightbox behavior script
     */
    public function lightbox_script() {
        static $done = false;
        if ( $done ) return;
        $done = true;
        ?>
        <script id="ht5e-lightbox-inline-js">
        (function($){
            let fsEnabled = false;

            // Helper: toggle fullscreen
            function toggleFullscreen(el){
                const doc=document;
                if(!fsEnabled){
                    fsEnabled=true;
                    if(el.requestFullscreen) el.requestFullscreen();
                    else if(el.webkitRequestFullscreen) el.webkitRequestFullscreen();
                    else if(el.mozRequestFullScreen) el.mozRequestFullScreen();
                    else if(el.msRequestFullscreen) el.msRequestFullscreen();
                }else{
                    fsEnabled=false;
                    if(doc.exitFullscreen) doc.exitFullscreen();
                    else if(doc.webkitExitFullscreen) doc.webkitExitFullscreen();
                    else if(doc.mozCancelFullScreen) doc.mozCancelFullScreen();
                    else if(doc.msExitFullscreen) doc.msExitFullscreen();
                }
            }

            // Listen for fullscreen exit
            ['fullscreenchange','webkitfullscreenchange','mozfullscreenchange','MSFullscreenChange'].forEach(evt=>{
                document.addEventListener(evt,()=>{ if(!document.fullscreenElement) fsEnabled=false; });
            });

            // Open lightbox
            $(document).on('click','.ht5e-launch',function(e){
                e.preventDefault();
                const src=$(this).data('ht5e-src');
                const scroll=$(this).data('ht5e-scroll')||'auto';
                const $lb=$('#ht5e-lightbox');
                const $iframe=$lb.find('.ht5e-iframe');
                const $loader=$('#ht5e-loader');

                $loader.addClass('active');
                $iframe.css('opacity',0);

                $iframe
                    .attr({src:src,scrolling:scroll})
                    .one('load',function(){
                        $loader.removeClass('active');
                        $(this).css('opacity',1);
                    });

                $lb.fadeIn(150);
                $('body').addClass('ht5e-open');
            });

            // Close lightbox
            $(document).on('click','#ht5e-lightbox .ht5e-close,#ht5e-lightbox .ht5e-overlay',function(){
                const $lb=$('#ht5e-lightbox');
                $lb.fadeOut(150,function(){
                    $lb.find('.ht5e-iframe').attr('src','');
                    $('body').removeClass('ht5e-open');
                });
            });

            // Keyboard shortcuts
            $(document).on('keyup',function(e){
                if(e.key==='Escape') $('#ht5e-lightbox .ht5e-close').trigger('click');
                if(e.key.toLowerCase()==='f'){
                    const iframe=document.querySelector('#ht5e-lightbox .ht5e-iframe');
                    if(iframe) toggleFullscreen(iframe);
                }
            });

            // Add fullscreen button
            $(document).ready(function(){
                const $modal=$('#ht5e-lightbox .ht5e-modal');
                if($modal.length && !$modal.find('.ht5e-fullscreen').length){
                    const $btn=$('<button class="ht5e-fullscreen" title="Toggle Fullscreen" aria-label="Toggle Fullscreen">⛶</button>');
                    $btn.css({
                        position:'absolute',top:'12px',left:'16px',
                        background:'transparent',border:'none',
                        color:'#fff',fontSize:'22px',cursor:'pointer'
                    });
                    $modal.append($btn);
                    $btn.on('click',function(e){
                        e.preventDefault();
                        const iframe=$modal.find('.ht5e-iframe')[0];
                        if(iframe) toggleFullscreen(iframe);
                    });
                }
            });

            // Auto resize if postMessage sent from iframe
            window.addEventListener('message',function(event){
                if(event.data && event.data.ht5eHeight){
                    const iframes=document.querySelectorAll('.ht5e-iframe');
                    iframes.forEach((iframe)=>{
                        if(iframe.contentWindow===event.source)
                            iframe.style.height=event.data.ht5eHeight+'px';
                    });
                }
            });

        })(jQuery);
        </script>
        <?php
    }
}
