<?php
if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Admin interface for HT5 Embedder
 * – Upload HTML5 packages
 * – Manage settings (width, height, theme, debug)
 */
class HT5E_Admin {

    public function __construct() {
        add_action( 'admin_menu', [ $this, 'menu' ] );
        add_action( 'admin_enqueue_scripts', [ $this, 'enqueue' ] );
    }

    /**
     * Admin menus
     */
    public function menu() {
        add_menu_page(
            'HT5 Embedder',
            'HT5 Embedder',
            'upload_files',
            'ht5e_upload',
            [ $this, 'page_upload' ],
            'dashicons-format-video',
            25
        );

        add_submenu_page(
            'ht5e_upload',
            'Upload HTML5',
            'Upload HTML5',
            'upload_files',
            'ht5e_upload',
            [ $this, 'page_upload' ]
        );

        add_submenu_page(
            'ht5e_upload',
            'Settings',
            'Settings',
            'manage_options',
            'ht5e_settings',
            [ $this, 'page_settings' ]
        );
    }

    /**
     * Enqueue admin assets
     */
    public function enqueue( $hook ) {
        if ( strpos( $hook, 'ht5e' ) === false ) return;

        wp_enqueue_style( 'ht5e-admin', HT5E_URL . 'assets/css/admin.css', [], HT5E_VERSION );
        wp_enqueue_script( 'ht5e-uploader', HT5E_URL . 'assets/js/uploader-page.js', [ 'jquery' ], HT5E_VERSION, true );

        wp_localize_script( 'ht5e-uploader', 'HT5EUpload', [
            'ajax'  => admin_url( 'admin-ajax.php' ),
            'nonce' => wp_create_nonce( 'ht5e-upload' ),
            'debug' => (bool) get_option( 'ht5e_debug', false ),
        ] );
    }

    /**
     * Upload HTML5 page
     */
    public function page_upload() { ?>
        <div class="wrap ht5e-admin">
            <h1>Upload HTML5 Package</h1>
            <p class="description">
                Upload a zipped HTML5 or SCORM package (Captivate, Storyline, iSpring, Camtasia, etc.).
                After upload, you’ll get a shortcode for embedding it in any page or post.
            </p>

            <table class="form-table">
                <tr>
                    <th><label for="ht5e-title">Title</label></th>
                    <td>
                        <input id="ht5e-title" class="regular-text" placeholder="My HTML5 Course">
                        <p class="description">This title will appear in the package list and shortcode reference.</p>
                    </td>
                </tr>
                <tr>
                    <th><label for="ht5e-file">ZIP Package (.zip)</label></th>
                    <td>
                        <input type="file" id="ht5e-file" accept=".zip">
                        <button id="ht5e-upload-btn" class="button button-primary">
                            <span class="dashicons dashicons-upload"></span> Upload ZIP
                        </button>
                        <span id="ht5e-status" class="ht5e-status"></span>
                        <p class="description">Allowed formats: HTML5/SCORM ZIP exports (must include <code>index.html</code> or similar).</p>
                    </td>
                </tr>
            </table>

            <!-- Upload result -->
            <div id="ht5e-result" style="display:none;margin-top:25px;">
                <h2>Package Uploaded Successfully</h2>
                <div class="ht5e-card">
                    <p><strong>Shortcode:</strong>
                        <code id="ht5e-shortcode"></code>
                        <button id="ht5e-copy" class="button button-secondary">Copy</button>
                        <span class="ht5e-copied" style="display:none;margin-left:6px;color:#16a34a;">Copied!</span>
                    </p>

                    <p><strong>Entry URL:</strong>
                        <a id="ht5e-entry" href="#" target="_blank" rel="noopener"></a>
                    </p>

                    <p>
                        <button id="ht5e-preview" class="button button-secondary">
                            <span class="dashicons dashicons-visibility"></span> Preview Content
                        </button>
                    </p>
                </div>
            </div>

            <!-- Admin lightbox preview -->
            <div id="ht5e-admin-lightbox" style="display:none;">
                <div class="ht5e-overlay"></div>
                <div class="ht5e-modal">
                    <button class="ht5e-close" aria-label="Close">&times;</button>
                    <div class="ht5e-frame ht5e-aspect ht5e-aspect--16x9">
                        <iframe class="ht5e-iframe" src="" allow="fullscreen" style="width:100%;height:100%;border:0;"></iframe>
                    </div>
                </div>
            </div>
        </div>
    <?php }

    /**
     * Settings page
     */
    public function page_settings() {
        if ( isset( $_POST['ht5e_save_settings'] ) && check_admin_referer( 'ht5e_settings_nonce', 'ht5e_settings_nonce_field' ) ) {
            update_option( 'ht5e_default_width',  sanitize_text_field( $_POST['ht5e_default_width']  ?? '100%' ) );
            update_option( 'ht5e_default_height', sanitize_text_field( $_POST['ht5e_default_height'] ?? '600' ) );
            update_option( 'ht5e_default_mode',   sanitize_text_field( $_POST['ht5e_default_mode']   ?? 'inline' ) );
            update_option( 'ht5e_default_theme',  sanitize_text_field( $_POST['ht5e_default_theme']  ?? '01' ) );
            update_option( 'ht5e_debug',          isset( $_POST['ht5e_debug'] ) ? 1 : 0 );
            echo '<div class="updated"><p><strong>Settings saved.</strong></p></div>';
        }

        $width  = get_option( 'ht5e_default_width',  '100%' );
        $height = get_option( 'ht5e_default_height', '600' );
        $mode   = get_option( 'ht5e_default_mode',   'inline' );
        $theme  = get_option( 'ht5e_default_theme',  '01' );
        $debug  = get_option( 'ht5e_debug',          false );
        ?>
        <div class="wrap ht5e-settings">
            <h1>HT5 Embedder — Settings</h1>
            <form method="post" action="">
                <?php wp_nonce_field( 'ht5e_settings_nonce', 'ht5e_settings_nonce_field' ); ?>
                <table class="form-table">
                    <tr>
                        <th><label for="ht5e_default_width">Default Width</label></th>
                        <td>
                            <input type="text" id="ht5e_default_width" name="ht5e_default_width" value="<?php echo esc_attr( $width ); ?>" class="regular-text">
                            <p class="description">Example: <code>100%</code> or <code>960px</code></p>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="ht5e_default_height">Default Height</label></th>
                        <td>
                            <input type="text" id="ht5e_default_height" name="ht5e_default_height" value="<?php echo esc_attr( $height ); ?>" class="regular-text">
                            <p class="description">Height of embedded player (pixels).</p>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="ht5e_default_mode">Default Mode</label></th>
                        <td>
                            <select name="ht5e_default_mode" id="ht5e_default_mode">
                                <option value="inline"   <?php selected( $mode, 'inline' ); ?>>Inline (embed directly)</option>
                                <option value="lightbox" <?php selected( $mode, 'lightbox' ); ?>>Lightbox (open on click)</option>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="ht5e_default_theme">Default Theme</label></th>
                        <td>
                            <select name="ht5e_default_theme" id="ht5e_default_theme">
                                <?php for ( $i=1; $i<=13; $i++ ): $n=str_pad($i,2,'0',STR_PAD_LEFT); ?>
                                    <option value="<?php echo $n; ?>" <?php selected($theme,$n); ?>>Theme <?php echo $n; ?></option>
                                <?php endfor; ?>
                            </select>
                            <div style="margin-top:8px;">
                                <?php for ( $i=1; $i<=13; $i++ ): $n=str_pad($i,2,'0',STR_PAD_LEFT); ?>
                                    <span style="display:inline-block;width:22px;height:22px;margin:1px;border-radius:3px;background:var(--t<?php echo $n; ?>,#1e90ff);border:1px solid #ccc;"></span>
                                <?php endfor; ?>
                            </div>
                        </td>
                    </tr>
                    <tr>
                        <th>Debug Mode</th>
                        <td>
                            <label><input type="checkbox" name="ht5e_debug" value="1" <?php checked( $debug, 1 ); ?>> Enable detailed debug logs in console</label>
                        </td>
                    </tr>
                </table>
                <?php submit_button( 'Save Settings', 'primary', 'ht5e_save_settings' ); ?>
            </form>
        </div>
        <?php
    }
}
