/**
 * HT5 Embedder v1.9.5
 * Admin upload controller
 * Handles: title + ZIP upload + AJAX + loader + preview + shortcode copy
 */
(function ($) {
  'use strict';

  const ajaxUrl = HT5EUpload.ajax;
  const nonce = HT5EUpload.nonce;
  const debug = HT5EUpload.debug;

  const $title = $('#ht5e-title');
  const $file = $('#ht5e-file');
  const $btn = $('#ht5e-upload-btn');
  const $status = $('#ht5e-status');
  const $result = $('#ht5e-result');
  const $shortcode = $('#ht5e-shortcode');
  const $entry = $('#ht5e-entry');
  const $copy = $('#ht5e-copy');
  const $preview = $('#ht5e-preview');
  const $loader = $('<div class="ht5e-loader"><div class="ht5e-spinner"></div></div>').appendTo('body').hide();

  /** Debug helper */
  function log() {
    if (debug) console.log('[HT5E Upload]', ...arguments);
  }

  /** Reset UI state */
  function resetUI() {
    $status.removeClass('ht5e-success ht5e-error').text('');
    $result.hide();
    $loader.hide();
  }

  /** Copy shortcode to clipboard */
  $copy.on('click', function () {
    const text = $shortcode.text();
    navigator.clipboard.writeText(text).then(() => {
      $('.ht5e-copied').fadeIn(100).delay(1000).fadeOut(200);
    });
  });

  /** Preview button (opens admin lightbox) */
  $preview.on('click', function (e) {
    e.preventDefault();
    const url = $entry.attr('href');
    if (!url) return;

    const $lb = $('#ht5e-admin-lightbox');
    const $iframe = $lb.find('.ht5e-iframe');
    $iframe.attr('src', url);
    $lb.fadeIn(150);
  });

  /** Close admin lightbox */
  $(document).on('click', '#ht5e-admin-lightbox .ht5e-close, #ht5e-admin-lightbox .ht5e-overlay', function () {
    $('#ht5e-admin-lightbox').fadeOut(150, function () {
      $(this).find('.ht5e-iframe').attr('src', '');
    });
  });

  /** Upload button */
  $btn.on('click', function (e) {
    e.preventDefault();
    resetUI();

    const file = $file[0].files[0];
    const title = $title.val().trim();

    if (!file) {
      alert('Please select a ZIP file first.');
      return;
    }

    $status.text('Uploading...');
    $loader.show();

    const formData = new FormData();
    formData.append('action', 'ht5e_upload_zip');
    formData.append('nonce', nonce);
    formData.append('file', file);
    formData.append('ht5e_title', title);

    $.ajax({
      url: ajaxUrl,
      type: 'POST',
      data: formData,
      contentType: false,
      processData: false,
      xhr: function () {
        const xhr = $.ajaxSettings.xhr();
        if (xhr.upload) {
          xhr.upload.addEventListener('progress', function (e) {
            if (e.lengthComputable) {
              const pct = Math.round((e.loaded / e.total) * 100);
              $status.text(`Uploading... ${pct}%`);
            }
          });
        }
        return xhr;
      },
      success: function (resp) {
        $loader.hide();
        if (resp.success) {
          log('Upload success:', resp.data);

          const data = resp.data;
          $status.text('Upload complete ✅').addClass('ht5e-success');
          $shortcode.text(data.shortcode);
          $entry.attr('href', data.entry).text(data.entry);
          $result.slideDown(250);

          if (debug && data.attempts) console.table(data.attempts);
        } else {
          $status.text('Upload failed ❌').addClass('ht5e-error');
          log('Upload failed:', resp);
        }
      },
      error: function (xhr, textStatus) {
        $loader.hide();
        $status.text('Error: ' + textStatus).addClass('ht5e-error');
        log('AJAX error:', xhr.responseText);
      }
    });
  });

  /** ESC closes lightbox */
  $(document).on('keyup', function (e) {
    if (e.key === 'Escape') $('#ht5e-admin-lightbox .ht5e-close').trigger('click');
  });

  log('HT5E uploader initialized.');
})(jQuery);
