/**
 * HT5 Embedder v1.9.5
 * Frontend controller for inline and lightbox embeds.
 * Handles: loader, fullscreen, keyboard shortcuts, dynamic resize.
 */
(function ($) {
  'use strict';

  let fsEnabled = false;

  /** Toggle fullscreen for an element */
  function toggleFullscreen(el) {
    const doc = document;
    if (!fsEnabled) {
      fsEnabled = true;
      if (el.requestFullscreen) el.requestFullscreen();
      else if (el.webkitRequestFullscreen) el.webkitRequestFullscreen();
      else if (el.mozRequestFullScreen) el.mozRequestFullScreen();
      else if (el.msRequestFullscreen) el.msRequestFullscreen();
    } else {
      fsEnabled = false;
      if (doc.exitFullscreen) doc.exitFullscreen();
      else if (doc.webkitExitFullscreen) doc.webkitExitFullscreen();
      else if (doc.mozCancelFullScreen) doc.mozCancelFullScreen();
      else if (doc.msExitFullscreen) doc.msExitFullscreen();
    }
  }

  /** Track when fullscreen exits manually (ESC or user action) */
  ['fullscreenchange','webkitfullscreenchange','mozfullscreenchange','MSFullscreenChange'].forEach(evt => {
    document.addEventListener(evt, () => {
      if (
        !document.fullscreenElement &&
        !document.webkitFullscreenElement &&
        !document.mozFullScreenElement &&
        !document.msFullscreenElement
      ) fsEnabled = false;
    });
  });

  /** Launch lightbox when Launch button is clicked */
  $(document).on('click', '.ht5e-launch', function (e) {
    e.preventDefault();
    const src = $(this).data('ht5e-src');
    const scroll = $(this).data('ht5e-scroll') || 'auto';
    const $lb = $('#ht5e-lightbox');
    const $iframe = $lb.find('.ht5e-iframe');
    const $loader = $('#ht5e-loader');

    // Reset loader + iframe
    $loader.addClass('active');
    $iframe.css('opacity', 0);

    // Load new content
    $iframe
      .attr({ src: src, scrolling: scroll })
      .one('load', function () {
        $loader.removeClass('active');
        $(this).css('opacity', 1);
      });

    $lb.fadeIn(150);
    $('body').addClass('ht5e-open');
  });

  /** Close lightbox overlay */
  $(document).on('click', '#ht5e-lightbox .ht5e-close, #ht5e-lightbox .ht5e-overlay', function () {
    const $lb = $('#ht5e-lightbox');
    $lb.fadeOut(150, function () {
      $lb.find('.ht5e-iframe').attr('src', '');
      $('body').removeClass('ht5e-open');
    });
  });

  /** Keyboard shortcuts: ESC closes, F toggles fullscreen */
  $(document).on('keyup', function (e) {
    if (e.key === 'Escape') $('#ht5e-lightbox .ht5e-close').trigger('click');
    else if (e.key.toLowerCase() === 'f') {
      const iframe = document.querySelector('#ht5e-lightbox .ht5e-iframe');
      if (iframe) toggleFullscreen(iframe);
    }
  });

  /** Add fullscreen button dynamically */
  $(document).ready(function () {
    const $modal = $('#ht5e-lightbox .ht5e-modal');
    if ($modal.length && !$modal.find('.ht5e-fullscreen').length) {
      const $btn = $('<button class="ht5e-fullscreen" title="Toggle Fullscreen" aria-label="Toggle Fullscreen">⛶</button>');
      $btn.css({
        position: 'absolute',
        top: '12px',
        left: '16px',
        background: 'transparent',
        border: 'none',
        color: '#fff',
        fontSize: '22px',
        cursor: 'pointer',
        zIndex: 9999
      });
      $modal.append($btn);
      $btn.on('click', function (e) {
        e.preventDefault();
        const iframe = $modal.find('.ht5e-iframe')[0];
        if (iframe) toggleFullscreen(iframe);
      });
    }
  });

  /** Auto-resize inline iframes based on postMessage */
  window.addEventListener('message', function (event) {
    if (event.data && event.data.ht5eHeight) {
      document.querySelectorAll('.ht5e-iframe').forEach((iframe) => {
        if (iframe.contentWindow === event.source)
          iframe.style.height = event.data.ht5eHeight + 'px';
      });
    }
  });

  console.log('%cHT5 Embedder frontend initialized', 'color:#1e90ff');

})(jQuery);
