<?php
/**
 * Plugin Name: DSpace-WordPress Connector
 * Description: Manage multiple DSpace repositories, display collections/items, and visualize relationships. Includes dropdowns populated from API + Test Connection button + Repo status indicators with Recheck button + Tooltip with last check info.
 * Version: 6.4
 * Author: Dr Saptarshi Ghosh & OpenAI
 */

// Register settings
function dspace_register_settings() {
    add_option('dspace_repositories', array());
    add_option('dspace_default_repo', '');
    add_option('dspace_default_community', '');
    add_option('dspace_default_collection', '');
    add_option('dspace_items_per_page', 5);

    register_setting('dspace_settings_group', 'dspace_repositories');
    register_setting('dspace_settings_group', 'dspace_default_repo');
    register_setting('dspace_settings_group', 'dspace_default_community');
    register_setting('dspace_settings_group', 'dspace_default_collection');
    register_setting('dspace_settings_group', 'dspace_items_per_page');
}
add_action('admin_init', 'dspace_register_settings');

// Add menu
function dspace_register_options_page() {
    add_menu_page('DSpace Settings', 'DSpace', 'manage_options', 'dspace-settings', 'dspace_options_page_html', 'dashicons-database', 100);
}
add_action('admin_menu', 'dspace_register_options_page');

// Settings page HTML
function dspace_options_page_html() {
    if (!current_user_can('manage_options')) return;
    $repos = get_option('dspace_repositories', array());
    $default_repo = get_option('dspace_default_repo', '');
    ?>
    <div class="wrap">
      <h1>DSpace Repositories</h1>
      <form method="post" action="options.php">
        <?php settings_fields('dspace_settings_group'); ?>
        <?php do_settings_sections('dspace_settings_group'); ?>

        <h2>Defaults</h2>
        <p>
          <label>Default Repo:
            <select name="dspace_default_repo" id="dspace_default_repo">
              <?php foreach ($repos as $slug => $repo): ?>
                <option value="<?php echo esc_attr($slug); ?>"
                        data-api="<?php echo esc_attr($repo['url']); ?>"
                        <?php selected($slug, $default_repo); ?>>
                  <?php echo esc_html($repo['name']); ?>
                </option>
              <?php endforeach; ?>
            </select>
          </label>
        </p>
        <p>
          <button type="button" id="recheck-repos" class="button">Recheck Repos</button>
        </p>

        <p>
          <label>Default Community:
            <select name="dspace_default_community" id="dspace_default_community">
              <option value="">-- Select Community --</option>
            </select>
          </label>
        </p>

        <p>
          <label>Default Collection:
            <select name="dspace_default_collection" id="dspace_default_collection">
              <option value="">-- Select Collection --</option>
            </select>
          </label>
        </p>

        <p>
          <label>Items per Page:
            <input type="number" name="dspace_items_per_page"
                   value="<?php echo esc_attr(get_option('dspace_items_per_page', 5)); ?>"
                   min="1" max="100">
          </label>
        </p>

        <p>
          <button type="button" id="test-connection" class="button">Test Connection</button>
          <span id="test-result" style="margin-left:10px;"></span>
        </p>

        <?php submit_button(); ?>
      </form>
    </div>
    <script>
    jQuery(document).ready(function($) {
        function loadCommunities() {
            let repoSelect = $('#dspace_default_repo');
            let apiUrl = repoSelect.find(':selected').data('api');
            if (!apiUrl) return;
            $.post(ajaxurl, {
                action: 'dspace_fetch_communities',
                api_url: apiUrl
            }, function(resp) {
                let $comm = $('#dspace_default_community');
                $comm.empty().append('<option value="">-- Select Community --</option>');
                if (resp.success) {
                    resp.data.forEach(function(c) {
                        let sel = (c.uuid === '<?php echo get_option('dspace_default_community'); ?>') ? 'selected' : '';
                        $comm.append('<option value="'+c.uuid+'" '+sel+'>'+c.name+'</option>');
                    });
                }
            });
        }
        function loadCollections(communityId) {
            let repoSelect = $('#dspace_default_repo');
            let apiUrl = repoSelect.find(':selected').data('api');
            if (!apiUrl || !communityId) return;
            $.post(ajaxurl, {
                action: 'dspace_fetch_collections',
                api_url: apiUrl,
                community_id: communityId
            }, function(resp) {
                let $coll = $('#dspace_default_collection');
                $coll.empty().append('<option value="">-- Select Collection --</option>');
                if (resp.success) {
                    resp.data.forEach(function(c) {
                        let sel = (c.uuid === '<?php echo get_option('dspace_default_collection'); ?>') ? 'selected' : '';
                        $coll.append('<option value="'+c.uuid+'" '+sel+'>'+c.name+'</option>');
                    });
                }
            });
        }
        $('#dspace_default_repo').on('change', function(){ loadCommunities(); });
        $('#dspace_default_community').on('change', function(){ loadCollections($(this).val()); });
        loadCommunities();
        let currentCommunity = '<?php echo get_option('dspace_default_community'); ?>';
        if (currentCommunity) loadCollections(currentCommunity);

        // Test Connection button
        $('#test-connection').on('click', function() {
            let apiUrl = $('#dspace_default_repo option:selected').data('api');
            if (!apiUrl) {
                $('#test-result').text('⚠️ Please select a repo first');
                return;
            }
            $('#test-result').text('⏳ Testing...');
            $.post(ajaxurl, {
                action: 'dspace_test_connection',
                api_url: apiUrl
            }, function(resp) {
                if (resp.success) {
                    $('#test-result').text(resp.data).css('color','green');
                } else {
                    $('#test-result').text(resp.data || 'Failed').css('color','red');
                }
            }).fail(function() {
                $('#test-result').text('❌ Error during request').css('color','red');
            });
        });

        // Repo status checker with tooltip
        function checkRepos() {
            $('#dspace_default_repo option').each(function() {
                let $opt = $(this);
                let apiUrl = $opt.data('api');
                if (!apiUrl) return;
                let baseText = $opt.text().replace(/ ✅.*| ❌.*/g, '');
                $.post(ajaxurl, {
                    action: 'dspace_test_connection',
                    api_url: apiUrl
                }, function(resp) {
                    let now = new Date().toLocaleString();
                    if (resp.success) {
                        $opt.text(baseText + ' ✅').attr('title', 'Last check: '+now+'\n'+resp.data);
                    } else {
                        $opt.text(baseText + ' ❌').attr('title', 'Last check: '+now+'\n'+(resp.data || 'Failed'));
                    }
                });
            });
        }

        // Auto-run on page load
        checkRepos();

        // Run on recheck button click
        $('#recheck-repos').on('click', function() {
            checkRepos();
        });
    });
    </script>
    <?php
}

// AJAX endpoints
function dspace_fetch_communities() {
    $api_url = esc_url_raw($_POST['api_url'] ?? '');
    if (!$api_url) wp_send_json_error("Missing API URL");
    $response = wp_remote_get($api_url . '/communities', ['timeout' => 20]);
    if (is_wp_error($response)) wp_send_json_error("Error fetching communities");
    $data = json_decode(wp_remote_retrieve_body($response), true);
    wp_send_json_success($data);
}
add_action('wp_ajax_dspace_fetch_communities', 'dspace_fetch_communities');

function dspace_fetch_collections() {
    $api_url = esc_url_raw($_POST['api_url'] ?? '');
    $community_id = sanitize_text_field($_POST['community_id'] ?? '');
    if (!$api_url || !$community_id) wp_send_json_error("Missing parameters");
    $url = $api_url . '/communities/' . $community_id . '/collections';
    $response = wp_remote_get($url, ['timeout' => 20]);
    if (is_wp_error($response)) wp_send_json_error("Error fetching collections");
    $data = json_decode(wp_remote_retrieve_body($response), true);
    wp_send_json_success($data);
}
add_action('wp_ajax_dspace_fetch_collections', 'dspace_fetch_collections');

// AJAX: Test connection to API
function dspace_test_connection() {
    $api_url = esc_url_raw($_POST['api_url'] ?? '');
    if (!$api_url) wp_send_json_error("Missing API URL");

    $response = wp_remote_get($api_url . '/status', ['timeout' => 15]);
    if (is_wp_error($response)) {
        wp_send_json_error("Error: " . $response->get_error_message());
    }

    $code = wp_remote_retrieve_response_code($response);
    $body = wp_remote_retrieve_body($response);

    if ($code == 200 && $body) {
        wp_send_json_success("HTTP 200 OK (" . esc_html($api_url) . ")");
    } else {
        wp_send_json_error("Failed with HTTP " . $code);
    }
}
add_action('wp_ajax_dspace_test_connection', 'dspace_test_connection');
?>
